function [ spaceTransformationFactor ] = computespacetransformationfactor( constrainedDistanceMat, discretePositionIndToCoordsMap, spatialBinWidth )
    % Here we compute the matrix of factors required to transform Euclidean
    % distance components (displacement in one dimension) to transformed
    % space coordinates.
    %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % validDiscretePositionsBinMat - yDim by xDim matrix, where yDim, xDim
    % are the vertical and horizontal spatial discretisation dimensions.
    %
    % constrainedDistanceMat - full constrained distance matrix (not upper
    % triangular or sparse).
    %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % CHANGELOG
    % 23.10.2014 - created, coded functionality.
    %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    if nargin < 3,
        error('Insufficient arguments!')
    end
    nDiscretePositions = size(constrainedDistanceMat, 1);
    if size(constrainedDistanceMat, 2) ~= nDiscretePositions,
        error('constrainedDistanceMat must be square!')
    end
    if size(discretePositionIndToCoordsMap, 1) ~= nDiscretePositions,
        error('Size of discretePositionIndToCoordsMap does not match constrainedDistanceMat!')
    end
    if size(discretePositionIndToCoordsMap, 2) ~= 2,
        error('discretePositionIndToCoordsMap should have two columns!')
    end
    
    % Compute the Euclidean distance between each discrete position.
    % We setup two lists of all discrete positions: 'sources' and
    % 'destinations', matched up in all possible pairings (without
    % repetition).
   
    euclideanDistanceMat = zeros([nDiscretePositions, nDiscretePositions]);
    coordsMat = repmat((1:nDiscretePositions)', [1, nDiscretePositions]);
    
    sourceInds = tril(coordsMat, -1);
    sourceInds = sourceInds(sourceInds > 0);
    sourceCoords = discretePositionIndToCoordsMap(sourceInds, :);
    
    destinationInds = triu(coordsMat, 1)';
    destinationInds = destinationInds(destinationInds > 0);
    destinationCoords = discretePositionIndToCoordsMap(destinationInds, :);
    
    euclideanDistanceVec = computeeuclideandistance( sourceCoords, destinationCoords );
    binInds = coordsMat > coordsMat';
    % Upper triangular matrix.
    euclideanDistanceMat(binInds) = euclideanDistanceVec * spatialBinWidth;
    euclideanDistanceMat = euclideanDistanceMat + euclideanDistanceMat';

    spaceTransformationFactor = constrainedDistanceMat ./ euclideanDistanceMat;
    
    % We must set diagonal to 1 to complete transformation factor.
    diagInds = (1:nDiscretePositions) + (nDiscretePositions * (0:(nDiscretePositions - 1)));
    spaceTransformationFactor(diagInds) = 1;
end