function [gridCentreCoordsMat, positionIndToGridCentreCoordsMap, gridCentreCoordsTrajectory] = formgridcentrecoords( validRegionsBinMat, discreteCoordsTrajectory, spatialBinWidth, minYCoord, minXCoord, yDim, xDim, nValidPositions )
    % Return the coordinates (pixels/continuous space) of the centre points
    % of the discrete grid cells that cover the entire maze. This includes
    % all accessible positions - those visited or unvisited yet implied to
    % be part of the maze ('valid' positions). We return a matrix of these
    % coordinates, and a trajectory matrix of these coordinates as they are
    % observed. We also return a matrix of mappings from linear position
    % indices to grid coordinates.
    %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % validRegionsBinMat - the logical binary matrix that indicates whether
    % discrete positions are considered valid/accessible.
    %
    % discreteCoordsTrajectory - observed trajectory of discrete positions.
    % Must be zeroed so that the minimum X and Y are both 1.
    %
    % spatialBinWidth - width of a discrete grid cell, in pixels.
    %
    % minYCoord, minXCoord - what should we use to convert the discrete
    % coordinates back to continuous-space coordinates?: the coordinate of
    % the lower left corner of the lower left discrete grid cell. Should be
    % the floor of the minimum observed coordinate.
    %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % CHANGELOG
    % 30.09.2014 - created, coded.
    % 26.10.2014 - added positionIndToGridCentreCoordsMap to output.
    % 29.10.2014 - added minYCoord, minXCoord functionality.
    %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    if nargin < 5,
        error('Insufficient arguments!')
    end
    if nargin < 7,
        yDim = size(validRegionsBinMat, 1);
        xDim = size(validRegionsBinMat, 2);
    end
    if nargin < 8,
        nValidPositions = sum(sum(validRegionsBinMat));
    else
        if nValidPositions ~= sum(sum(validRegionsBinMat)),
            error('nValidPositions does not match the valid positions in validPositionsLogMat!')
        end
    end

    % The first page stores the Y-dimension centre coordinate; the second
    % page stores the corresponding X-dimension coordinate.
    gridCentreCoordsMat = nan([size(validRegionsBinMat), 2]);
    gridCentreCoordsMat(:, :, 1) = repmat((1:yDim)', [1, xDim]);
    gridCentreCoordsMat(:, :, 2) = repmat(1:xDim, [yDim, 1]);
    gridCentreCoordsMat = gridCentreCoordsMat * spatialBinWidth - spatialBinWidth / 2;
    gridCentreCoordsMat(:, :, 1) = gridCentreCoordsMat(:, :, 1) + minYCoord;
    gridCentreCoordsMat(:, :, 2) = gridCentreCoordsMat(:, :, 2) + minXCoord;
    gridCentreCoordsMat(~repmat(validRegionsBinMat, [1, 1, 2])) = nan;
    
    positionIndToGridCentreCoordsMap = reshape(gridCentreCoordsMat(repmat(validRegionsBinMat, [1, 1, 2])), [nValidPositions, 2]);
    
    gridCentreCoordsTrajectory = discreteCoordsTrajectory * spatialBinWidth - spatialBinWidth / 2;
    gridCentreCoordsTrajectory = bsxfun(@plus, gridCentreCoordsTrajectory, [minYCoord, minXCoord]);
end