function linearisedTrajectory = formlinearisedtrajectory( coordsTrajectory, validPositionsLogMat, yDim, xDim )
    % Take a matrix of discretised position observations and the grid of
    % logical indicators specifying where (and how many) discrete positions
    % are, and return the corresponding matrix of linearised positions,
    % i.e. the linear indices that would access those positions from the
    % grid if only valid positions were counted.
    %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % coordsTrajectory - an n by 2 matrix of positive integers. These are
    % the processed, discretised observed positions on n time bins.
    %
    % linearisedTrajectory - an n by 1 matrix of positive integers, the
    % linear indices of observed valid positions. These count the valid
    % positions in linear indexing fashion. E.g., if the X = 1 column
    % contains valid positions at (2, 1), (3, 1) and (6, 1) only, then
    % these three positions are the linearised positions 1, 2 and 3
    % respectively. Then if X = 2 column has valid positions at (1, 2), (3,
    % 2) and (4, 2) only, these are linearised positions 4, 5 and 6
    % respectively.
    %
    % validPositionsLogMat - a 2D matrix with number of rows equal to the
    % Y-dimension size of the discretisation grid and number of columns
    % equal to the X-dimension size of the discretisation grid. It contains
    % logicals only.
    %
    % yDim, xDim - the size of the discretisation grid we use in the Y- and
    % X-dimensions respectively; i.e. the number of grid squares that cover
    % the environment in each dimension.
    %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % CHANGELOG
    % 29.09.2014 - copied from old version.
    %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    if nargin < 2,
        error('Insufficient arguments!')
    end
    if nargin < 4,
        dims = size(validPositionsLogMat);
        yDim = dims(1);
        xDim = dims(2);
        if max(coordsTrajectory(:, 1)) > yDim,
            error('Y dimension of coordsTrajectory exceeds that of validPositionsLogMat!');
        end
        if max(coordsTrajectory(:, 2)) > xDim,
            error('X dimension of coordsTrajectory exceeds that of validPositionsLogMat!');
        end
    else
        if rem(yDim, 1) ~= 0,
            error('yDim must be a positive integer!')
        end
        if yDim < 1,
            error('yDim must be a positive integer!')
        end
        if rem(xDim, 1) ~= 0,
            error('xDim must be a positive integer!')
        end
        if xDim < 1,
            error('xDim must be a positive integer!')
        end
        if max(coordsTrajectory(:, 1)) > yDim,
            error('Y dimension of coordsTrajectory exceeds yDim!');
        end
        if max(coordsTrajectory(:, 2)) > xDim,
            error('X dimension of coordsTrajectory exceeds xDim!');
        end
        if size(validPositionsLogMat) ~= [yDim, xDim],
            error('Dimensions of validPositionsLogMat do not match xDim, yDim!')
        end
    end

    nValidPos = sum(sum(validPositionsLogMat));
    
    linearIndsMat = zeros([yDim, xDim]);
    
    linearIndsMat(validPositionsLogMat) = (1:nValidPos)';

    absoluteLinearInds = sub2ind([yDim, xDim], coordsTrajectory(:, 1), coordsTrajectory(:, 2));

    if any(~validPositionsLogMat(absoluteLinearInds)),
        error('validPositionsLogMat seems to suggest some of the positions in coordsTrajectory are invalid!')
    end
    
    linearisedTrajectory = linearIndsMat(absoluteLinearInds);
    
    if size(linearisedTrajectory, 2) > 1,
        linearisedTrajectory = linearisedTrajectory';
    end
end