#include <stddef.h>
#include <stdlib.h>
#include <math.h>
#include <stdio.h>
#include <curses.h>

/* These are the definitions from the WaveThresh package. */

#define PERIODIC        1
#define SYMMETRIC       2

/* Increasing this value would remove some nearly empty diagonals */

double thr=0.0;

/* The doubledouble structure is used by the makegrid function for
   sorting the data with respect to the x-component. */

struct doubledouble
           {
		   double x;
		   double y;
		   } Doubledouble;

/* The ddcomp function is used by the makegrid function for sorting
   the data as an argument for qsort. */
		   
int ddcomp(struct doubledouble *q1, struct doubledouble *q2)
  {
  if(q1->x>q2->x)
    return 1;
  else
  if(q1->x<q2->x)
    return -1;
  else
    return 0;
  }

void makegrid(double *x,double *y,long *n,double *gridx,double *gridy,
               long *gridn, double *G, long *Gindex)

			   /* This function computes from observations in x und y
			      new data on a grid of length gridn as well as a
				  description of the matrix that maps the original
				  data (or better the ordered original data, so that
				  x[i]<=x[j] when i<=j) to the new grid data. 
			   
			      input:
			      x, y   vector of observations of length n 
				  gridn

				  output:
			      gridx, gridy the constructed grid of length gridn */
  {
  struct doubledouble *q;
  int i,li=0,ind;
  double dummy;

  /* First, sort the data with respect to x. */

  q=(void *)malloc(*n*sizeof(Doubledouble));

  for(i=0;i<*n;i++)
    {
    q[i].x=x[i];
    q[i].y=y[i];
    }
  qsort(q,*n,sizeof(Doubledouble),(int *)ddcomp);

  /* Now create the new grid data. */

  for(i=0;i<*gridn;i++)
    {
    gridx[i]=(i+0.5)/(*gridn);

    /* Determine the index of the nearest observation left to the grid time 
       point. */

    while((li<*n-1)&&(q[li+1].x<gridx[i]))
      li++;
    if(li==(*n-1)) /* We are at the right end */
	  {
      gridy[i]=q[li].y;
	  ind=li-1;
	  G[i]=0;
	  }
    else
    if(q[li].x>=gridx[i]) /* We are at the left end */
	  {
      gridy[i]=q[0].y;
	  ind=0;
	  G[i]=1;
	  }
    else
	  {
      gridy[i]=q[li].y+(gridx[i]-q[li].x)*(q[li+1].y-q[li].y)
                                                /(q[li+1].x-q[li].x); 
	  ind=li;
	  G[i]=1-(gridx[i]-q[li].x)/(q[li+1].x-q[li].x);
	  }
	Gindex[i]=ind;
    }
  free(q);
  }

/* sigmastruct describes a covariance matrix of size n. diag is an n-vector
   of pointers to double vectors that correspond to the diagonals of the
   matrix. If diag[i]==NULL, then the i-th diagonal is empty. This
   representation is useful for covariance matrices with a band structure. */

struct sigmastruct 
          {
		  int n;
          double **diag;
		  };

/* createSigma allocates memory for a new covariance matrix of size n. */

void createSigma(struct sigmastruct *Sigma, int n)
  {
  int i;
  
  Sigma->n=n;
  if((Sigma->diag=malloc(n*sizeof(double *)))==NULL)
    {
	puts("Couldn't allocate memory in createSigma");
	exit(1L);
	}
  for(i=0;i<n;i++)
    (Sigma->diag)[i]=NULL;
  }
  
/* freeSigma releases the memory used by a sigmastruct element. */

void freeSigma(struct sigmastruct *Sigma)
  {
  int i;

  for(i=0;i<Sigma->n;i++)
    if(Sigma->diag[i]!=NULL)
      free((Sigma->diag)[i]);
  free(Sigma->diag);
  }

/* CleanupSigma removes diagonals that contain only elements < thr. */
   
void cleanupSigma(struct sigmastruct *Sigma)
  {
  int i,j,zaehler;

  for(i=0;i<Sigma->n;i++)
    if((Sigma->diag)[i]!=NULL)
      {
      j=0;
      while((j<Sigma->n-i)&&(fabs((Sigma->diag)[i][j])<thr))
        j++;
      if(j>=Sigma->n-i)
        {
        free(Sigma->diag[i]);
        Sigma->diag[i]=NULL;
        }
      }
  }
  
/* putSigma changes the entry in the i-th row and j-th column to s and
   allocates memory for the diagonal if necessary. */

void putSigma(struct sigmastruct *Sigma, int i, int j, double s)
  {
  int d=abs(i-j),min;
  
  if(fabs(s)>0.0000001)
  if((i>=Sigma->n)||(j>=Sigma->n))
    {
	puts("Error: This element does not exist.");
	exit(1L);
	}
  else
    {
    if((Sigma->diag)[d]==NULL)
	  if((Sigma->diag[d]=calloc(Sigma->n-d,sizeof(double)))==NULL)
	    exit(1L);
    (Sigma->diag)[d][(i+j-d)/2]=s;
    }
  }

/* allocateSigma allocates memory for diagonals of a covariance matrix,
   specified by the boolean vector d */
   

void allocateSigma(struct sigmastruct *Sigma, int *d)
  {
  int i;
  
  for(i=0;i<Sigma->n;i++)
    if(d[i]==TRUE)
      if((Sigma->diag[i]=calloc(Sigma->n-i,sizeof(double)))==NULL)
        exit(1L);
  }

/* computec is the function that computes the factors for the variances
   of the wavelet coefficients. 

   Gmatrix, Gindex describe the matrix that maps the original data
                   to the grid data as received by makegrid (s.o.)
   n, gridn        are the numbers of original and grid observations
   H, LengthH      describe the used wavelet filter. The filter coefficients
                   are stored in the vector H, their number in LengthH.
   bc              is either PERIODIC or SYMMETRIC, the boundary correction
                   to be used

   c contains afterwards the coefficients c_{jk}, such that
               Var(w_{jk})=c_{jk}\cdot\sigma^2
*/


void computec(long *n,double *c,long *gridn,double *Gmatrix,long *Gindex,
              double *H, long *LengthH, long *bc)
  {
  int virtgn,i,j,k,l,d,testzaehler,zaehler=0,gn=*gridn,laststart=0;
  int ii,dd,jj,o1,o2,iiG,iiH,jjG,jjH,gn2,LengthH2=*LengthH/2,dH,dG;
  int *NEEDIT,offset,offset2,band,band1,band2;
  double *w,cell,cellC,cellD,sig,G[20];
  double ProductG[20][20],ProductH[20][20];
  char dummy[20];
  struct sigmastruct Sigma,Sigma2,Sigma3;

  if(*LengthH>20)
    {
    puts("Sorry, you can not use this procedure with more than 20 filter coefficients!");
    exit(1L); 
    }
  if((NEEDIT=malloc(*gridn*sizeof(int)))==NULL)
    exit(1L);

  /* First step: Compute Filter G from Filter H */
  sig=-1.0;
  for(k=0;k<*LengthH;k++)
    {
	G[*LengthH-k-1]=sig*H[k];
	sig=-sig;
	}

  for(k=0;k<*LengthH;k++)
    for(l=0;l<*LengthH;l++)
      {
      ProductG[k][l]=G[k]*G[l];
      ProductH[k][l]=H[k]*H[l];
      }

  /* Second step: Compute the variance/covariance-matrix of the grid data */

  createSigma(&Sigma,gn);
  for(i=0;i<*gridn;i++)
    {
	j=laststart;
	while(Gindex[i]-Gindex[j]>=2) 
	  j++;
	laststart=j;
	for(;j<=i;j++)
	  {
      switch(Gindex[i]-Gindex[j])
	    {
		case  1: putSigma(&Sigma,i,j,Gmatrix[i]*(1.0-Gmatrix[j]));
		         break;
		case  0: putSigma(&Sigma,i,j,Gmatrix[i]*Gmatrix[j]+(1.0-Gmatrix[i])*(1.0-Gmatrix[j]));
		         break;
		}
	  }
	}
  
  /* And now the difficult part... */
  
  if(*bc==PERIODIC)
    {
    while(gn>=2) /* Apply the wavelet filters to the covariance matrix Sigma. */
      {
      gn2=gn/2; /* gn and gn2 are the sizes of Sigma and Sigma2 (or Sigma3). */ 
      createSigma(&Sigma2,gn2); /* Store the result of the high pass filter in
                                   sigma2... */
      createSigma(&Sigma3,gn2); /* ... and the result of the low pass filter
      cleanupSigma(&Sigma);        in sigma3. */

      /* First we need to know which diagonals in sigma2 and sigma3 will not
         be empty. */

	  band1=gn/2;
	  band2=gn/2+1;
	  while((band1>=0)&&(Sigma.diag[band1]==NULL))
	    band1--;
	  while((band2<=gn-1)&&(Sigma.diag[band2]==NULL))
	    band2++;
      if(band1<=gn-band2)
	    band=gn-band2;
      else
	    band=band1;
		
      if(band+*LengthH>gn)
        for(d=0;d<gn2;d++)
	      NEEDIT[d]=TRUE;
      else
	    {
        for(d=0;d<gn2;d++)
	      NEEDIT[d]=FALSE;
        for(d=0;(d<=(band+*LengthH)/2)&&(d<gn2);d++)
          NEEDIT[d]=TRUE;
        for(d=1;(d<=(band+*LengthH)/2)&&(d<gn2);d++)
          NEEDIT[gn2-d]=TRUE;
	    }

      /* We allocate memory only for these diagonals. */

      allocateSigma(&Sigma2,NEEDIT);
      allocateSigma(&Sigma3,NEEDIT);

      /* Every entry of Sigma is involved in the computation of (LengthH/2)^2
         entries of Sigma2 and Sigma3. We find out in which and do the
         necessary computaions. */

      /* Let's start with the main diagonal elements of Sigma. */

      if(Sigma.diag[0]!=NULL)
	    for(j=0;j<gn;j++)
		  if(fabs(sig=Sigma.diag[0][j])>thr)
		    {
		    o1=j%2;
            iiH=j/2;
            iiG=(iiH+LengthH2-1)%(gn2);

		    for(k=0;k<LengthH2;k++)
			  {
              Sigma3.diag[0][iiH]+=sig*ProductH[2*k+o1][2*k+o1];
              Sigma2.diag[0][iiG]+=sig*ProductG[2*k+o1][2*k+o1];
              jjH=((j-2*(k+1)+*gridn)/2)%(gn2);
              jjG=(jjH+LengthH2-1)%(gn2);
		      for(l=k+1;l<LengthH2;l++)
			    { 
				dH=abs(iiH-jjH);
				dG=abs(iiG-jjG);
				if(dH==0)
                  Sigma3.diag[dH][iiH]+=2*sig*ProductH[2*k+o1][2*l+o1];
				else       
                  Sigma3.diag[dH][(iiH+jjH-dH)/2]+=sig*ProductH[2*k+o1][2*l+o1];
				if(dG==0)
                  Sigma2.diag[dG][iiG]+=2*sig*ProductG[2*k+o1][2*l+o1];
				else         
                  Sigma2.diag[dG][(iiG+jjG-dG)/2]+=sig*ProductG[2*k+o1][2*l+o1];
            
                if((--jjH)<0) jjH=gn2-1;
                if((--jjG)<0) jjG=gn2-1;
				}
              if((--iiH)<0) iiH=gn2-1;
              if((--iiG)<0) iiG=gn2-1;
		      }
            }

      /* Now do the rest. */
      for(d=1;d<gn;d++)
        if(Sigma.diag[d]!=NULL)
	      for(j=0;j<gn-d;j++)
		    if(fabs(sig=Sigma.diag[d][j])>thr)
		      {
			  o1=(d+j)%2;
              iiH=((d+j+*gridn)/2)%(gn2);
              iiG=(iiH+LengthH2-1)%(gn2);

		      for(k=0;k<LengthH2;k++)
			    {
                jjH=((j+*gridn)/2)%(gn2);
                jjG=(jjH+LengthH2-1)%(gn2);
			    o2=j%2;
		        for(l=0;l<LengthH2;l++)
				  {
				  dH=abs(iiH-jjH);
				  dG=abs(iiG-jjG);
				  if(dH==0)
                    Sigma3.diag[dH][iiH]+=2*sig*ProductH[o1][o2];
				  else       
                    Sigma3.diag[dH][(iiH+jjH-dH)/2]+=sig*ProductH[o1][o2];
				  if(dG==0)
                    Sigma2.diag[dG][iiG]+=2*sig*ProductG[o1][o2];
				  else         
                    Sigma2.diag[dG][(iiG+jjG-dG)/2]+=sig*ProductG[o1][o2];
                  if((--jjH)<0) jjH=gn2-1;
                  if((--jjG)<0) jjG=gn2-1;
                  o2+=2;
                  }
                if((--iiH)<0) iiH=gn2-1;
                if((--iiG)<0) iiG=gn2-1;
				o1+=2;
				}
              }

      /* Now Sigma2 und Sigma3 should contain the right values. Store
         only the main diagonal of Sigma2 in c. */
	  for(j=0;j<gn2;j++)
        c[zaehler++]=Sigma2.diag[0][j];

      /* Sigma and Sigma2 are now redundant. */
      freeSigma(&Sigma);
      freeSigma(&Sigma2);

      /* Sigma3 becomes our new Sigma. */
      memcpy(&Sigma,&Sigma3,sizeof(Sigma));
  	  gn=gn2;
  	  }
	} /* end of the periodic case */
  else
  if(*bc==SYMMETRIC) /* The symmetric case is even more difficult... */
    {
    virtgn=gn; /* virtgn is used to determine when we can finish and takes
                  exactly the same values as gn in the periodic case. */
    offset=0; /* Symmetric boundary conditions provide extra coefficients
                 on both sides. offset gives the number of such extra
                 coeffs on the left-hand side. */
    while(virtgn>=2)
      {
      /* First of all, we want to know how many diagonals and extra coeffs
         Sigma2 and Sigma3 have. */

	  if(offset%2==0)
	    gn2=(gn+1)/2+LengthH2-1;
      else
	    gn2=(gn+2)/2+LengthH2-1;
	  offset2=(offset+*LengthH-1)/2;

      /* Now allocate memory for them. */
	  
      createSigma(&Sigma2,gn2);
      createSigma(&Sigma3,gn2);
      cleanupSigma(&Sigma);
      
      /* Again, we need to know which diagonals in sigma2 and sigma3 will not
         be empty. */

	  band=gn-1;
	  while((band>=0)&&(Sigma.diag[band]==NULL))
	    band--;
      if(band+2*(*LengthH)>gn)
        for(d=0;d<gn2;d++)
	      NEEDIT[d]=TRUE;
      else
	    {
        for(d=0;d<gn2;d++)
  	      NEEDIT[d]=FALSE;
        for(d=0;(d<=(band+*LengthH)/2)&&(d<gn2);d++)
          NEEDIT[d]=TRUE;
        }

      /* We allocate memory only for these diagonals. */

      allocateSigma(&Sigma2,NEEDIT);
      allocateSigma(&Sigma3,NEEDIT);

      /* Every entry of Sigma is involved in the computation of (LengthH/2)^2
         entries of Sigma2 and Sigma3. We find out in which and do the
         necessary computaions. */

      /* Let's start with the values that are not effected by boundary 
         correction.  First the main diagonal. */

      if(Sigma.diag[0]!=NULL)
	    for(j=0;j<gn;j++)
		  if(fabs(sig=Sigma.diag[0][j])>thr)
		    {
		    o1=(j+offset)%2;
            iiH=(j+offset%2)/2+LengthH2-1;
			iiG=iiH;

		    for(k=0;k<LengthH2;k++)
			  {
              Sigma3.diag[0][iiH]+=sig*ProductH[2*k+o1][2*k+o1];
              Sigma2.diag[0][iiG]+=sig*ProductG[2*k+o1][2*k+o1];
              jjH=(j+offset%2)/2+LengthH2-1-(k+1);
			  jjG=jjH;
		      for(l=k+1;l<LengthH2;l++)
			    { 
				if((iiH<0)||(jjH<0))
				  puts("ERROR");
				dH=abs(iiH-jjH);
				dG=abs(iiG-jjG);
				if(dH==0)
                  Sigma3.diag[dH][iiH]+=2*sig*ProductH[2*k+o1][2*l+o1];
				else       
                  Sigma3.diag[dH][(iiH+jjH-dH)/2]+=sig*ProductH[2*k+o1][2*l+o1];
				if(dG==0)
                  Sigma2.diag[dG][iiG]+=2*sig*ProductG[2*k+o1][2*l+o1];
				else       
                  Sigma2.diag[dG][(iiG+jjG-dG)/2]+=sig*ProductG[2*k+o1][2*l+o1];
                jjH--;
                jjG--;
				}
              iiH--;
              iiG--;
		      }
            }

      /* Now do the not-effected values on the other diagonals. */

      for(d=1;d<gn;d++)
        if(Sigma.diag[d]!=NULL)
	      /*for(j=*LengthH-2+offset%2;j<gn-d-(*LengthH-2+offset%2);j++)*/
	      for(j=0;j<gn-d;j++)
		    if(fabs(sig=Sigma.diag[d][j])>thr)
		      {
		      o1=(d+j+offset)%2;
              iiH=(d+j+offset%2)/2+LengthH2-1;
			  iiG=iiH;

		      for(k=0;k<LengthH2;k++)
			    {
                jjH=(j+offset%2)/2+LengthH2-1;
			    jjG=jjH;
			    o2=(j+offset)%2;
		        for(l=0;l<LengthH2;l++)
				  {
				  dH=abs(iiH-jjH);
				  dG=abs(iiG-jjG);
				  if(dH==0)
                    Sigma3.diag[dH][iiH]+=2*sig*ProductH[o1][o2];
				  else       
                    Sigma3.diag[dH][(iiH+jjH-dH)/2]+=sig*ProductH[o1][o2];
				  if(dG==0)
                    Sigma2.diag[dG][iiG]+=2*sig*ProductG[o1][o2];
				  else         
                    Sigma2.diag[dG][(iiG+jjG-dG)/2]+=sig*ProductG[o1][o2];
                  jjH--;
                  jjG--;
                  o2+=2;
                  }
				iiH--;
				iiG--;
				o1+=2;
				}
              }

      /* Finally, do the boundary entries. */

      for(d=0;d<gn2;d++)
        if(Sigma2.diag[d]!=NULL)
		  {
		  for(j=0;(j<gn2-d)&&(j<LengthH2-1+offset%2);j++)
		    {
			cellC=0;
			cellD=0;
			i=d+j;
			iiG=2*i-(*LengthH-2)-offset%2;
		    for(k=0;k<*LengthH;k++,iiG++)
              {
		      jjG=2*j-(*LengthH-2)-offset%2;
		      for(l=0;l<*LengthH;l++,jjG++)
			    {
                ii=iiG;
                jj=jjG;
				if(ii<0)
				  ii=-ii-1;
				if(jj<0)
				  jj=-jj-1;
				if(ii>=gn)
                  {
				  ii=2*gn-ii-1;
			      }
				if(jj>=gn)
                  {
				  jj=2*gn-jj-1;
			      }
				dd=abs(ii-jj);
                if(Sigma.diag[dd]!=NULL)
                  {
				  sig=(Sigma.diag)[dd][(ii+jj-dd)/2];
				  cellC+=sig*ProductH[k][l];
				  cellD+=sig*ProductG[k][l];
                  }
				}
              }
            Sigma2.diag[d][j]=cellD;
            Sigma3.diag[d][j]=cellC;
			}
		  for(j=gn2-d-1;(j>=0)&&(j>=gn2-d-LengthH2+1-offset%2);j--)
		    {
			cellC=0;
			cellD=0;
			i=d+j;
			iiG=2*i-(*LengthH-2)-offset%2;
		    for(k=0;k<*LengthH;k++,iiG++)
              {
		      jjG=2*j-(*LengthH-2)-offset%2;
		      for(l=0;l<*LengthH;l++,jjG++)
			    {
                ii=iiG;
                jj=jjG;
				if(ii<0)
				  ii=-ii-1;
				if(jj<0)
				  jj=-jj-1;
				if(ii>=gn)
                  {
				  ii=2*gn-ii-1;
			      }
				if(jj>=gn)
                  {
				  jj=2*gn-jj-1;
			      }
				dd=abs(ii-jj);
                if(Sigma.diag[dd]!=NULL)
                  {
				  sig=(Sigma.diag)[dd][(ii+jj-dd)/2];
				  cellC+=sig*ProductH[k][l];
				  cellD+=sig*ProductG[k][l];
                  }
				}
              }
            Sigma2.diag[d][j]=cellD;
            Sigma3.diag[d][j]=cellC;
			}
		  }

      /* This looks now pretty the same as in the periodic case. */

	  for(j=0;j<gn2;j++)
        c[zaehler++]=Sigma2.diag[0][j];
      freeSigma(&Sigma);
      freeSigma(&Sigma2);
      memcpy(&Sigma,&Sigma3,sizeof(Sigma));

      gn=gn2;
	  offset=offset2;
      virtgn=virtgn/2;
      }
	} /* end of the symmetric case */

  /* We are finished with the computation of c and can release any memory. */

  freeSigma(&Sigma);
  free(NEEDIT);
  }

void makegrid2(double *x,double *y,long *n,double *gridx,double *gridy,
               long *gridn, double *G, long *Gindex, long *observn)

			   /* This function computes from observations in x und y
			      new data on a grid of length gridn as well as a
				  description of the matrix that maps the original
				  data (or better the ordered original data, so that
				  x[i]<=x[j] when i<=j) to the new grid data. 
			   
			      input:
			      x, y   vector of observations of length n 
				  gridn

				  output
			      gridx, gridy the constructed grid of length gridn */
  {
  struct doubledouble *q;
  int i,li=0,ind,left,right,newn;
  double dummy,sum;

  q=(void *)malloc(*n*sizeof(Doubledouble));

  for(i=0;i<*n;i++)
    {
    q[i].x=x[i];
    q[i].y=y[i];
    }
  qsort(q,*n,sizeof(Doubledouble),(void *)ddcomp);

  left=0;
  newn=0;
  while(left<*n)
    {
	right=left;
	while((right+1<*n)&&(fabs(q[right+1].x-q[left].x)<0.000001))
	  right++;
	sum=0;
	for(i=left;i<=right;i++)
	  sum+=q[i].y;
	x[newn]=q[left].x;
	y[newn]=sum/(right-left+1);
	observn[newn]=right-left+1;
	newn++;
	left=right+1;
	}
  *n=newn;
  for(i=0;i<*gridn;i++)
    {
    gridx[i]=(i+0.5)/(*gridn);
    while((li<*n-1)&&(x[li+1]<gridx[i]))
      li++;
    if(li==(*n-1)) /* We are at the right end */
	  {
      gridy[i]=y[li];
	  ind=li-1;
	  G[i]=0;
	  }
    else
    if(q[li].x>=gridx[i]) /* We are at the left end */
	  {
      gridy[i]=y[0];
	  ind=0;
	  G[i]=1;
	  }
    else
	  {
      gridy[i]=y[li]+(gridx[i]-x[li])*(y[li+1]-y[li])
                                                /(x[li+1]-x[li]); 
	  ind=li;
	  G[i]=1-(gridx[i]-x[li])/(x[li+1]-x[li]);
	  }
	Gindex[i]=ind;
    }
  free(q);
  }

void computec2(long *n,double *c,long *gridn,double *Gmatrix,long *Gindex,
               long *observn, double *H, long *LengthH, long *bc)
            /* Gmatrix, Gindex describe the matrix that maps the original data
               to the grid data as received by makegrid (s.o.)
               c contains the coefficients c_{jk}, so that
               Var(w_{jk})=c_{jk}\cdot\sigma^2
            */
  {
  int virtgn,i,j,k,l,d,testzaehler,zaehler=0,gn=*gridn,laststart=0;
  int ii,dd,jj,o1,o2,iiG,iiH,jjG,jjH,gn2,LengthH2=*LengthH/2,dH,dG;
  int *NEEDIT,offset,offset2,band,band1,band2;
  int iind, jind, iind1, jind1;
  double *w,cell,cellC,cellD,sig,G[20];
  double ProductG[20][20],ProductH[20][20];
  char dummy[20];
  struct sigmastruct Sigma,Sigma2,Sigma3;

  if(*LengthH>20)
    {
    puts("Sorry, you can not use this procedure with more than 20 filter coefficients!");
    exit(1L); 
    }
  if((NEEDIT=malloc(*gridn*sizeof(int)))==NULL)
    exit(1L);

  /* First step: Compute Filter G from Filter H */
  sig=-1.0;
  for(k=0;k<*LengthH;k++)
    {
	G[*LengthH-k-1]=sig*H[k];
	sig=-sig;
	}

  for(k=0;k<*LengthH;k++)
    for(l=0;l<*LengthH;l++)
      {
      ProductG[k][l]=G[k]*G[l];
      ProductH[k][l]=H[k]*H[l];
      }

  /* Second step: Compute the variance/covariance-matrix of the grid data */

  createSigma(&Sigma,gn);
  for(i=0;i<*gridn;i++)
    {
	iind=Gindex[i];
	if(iind<*n-1)
	  iind1=iind+1;
	else
	  iind1=iind;
	j=laststart;
	while(iind-Gindex[j]>=2) 
	  j++;
	laststart=j;
	for(;j<=i;j++)
	  {
	  jind=Gindex[j];
	  if(jind<*n-1)
	    jind1=jind+1;
	  else
	    jind1=jind;
      switch(iind-jind)
	    {
		case  1: putSigma(&Sigma,i,j,Gmatrix[i]*(1.0-Gmatrix[j])/(observn[iind]*observn[jind1]));
		         break;
		case  0: putSigma(&Sigma,i,j,Gmatrix[i]*Gmatrix[j]/(observn[iind]*observn[jind])+(1.0-Gmatrix[i])*(1.0-Gmatrix[j])/(observn[iind1]*observn[jind1]));
		         break;
		}
	  }
	}
  
/*
  reportSigma(&Sigma);
*/
  /* Compute the covariances */
  
  if(*bc==PERIODIC)
    {
    while(gn>=2)
      {
      gn2=gn/2;
      createSigma(&Sigma2,gn2);
      createSigma(&Sigma3,gn2);
      cleanupSigma(&Sigma);

      /* Compute actual bandwidth */

	  band1=gn/2;
	  band2=gn/2+1;
	  while((band1>=0)&&(Sigma.diag[band1]==NULL))
	    band1--;
	  while((band2<=gn-1)&&(Sigma.diag[band2]==NULL))
	    band2++;
      if(band1<=gn-band2)
	    band=gn-band2;
      else
	    band=band1;
		
      if(band+*LengthH>gn)
        for(d=0;d<gn2;d++)
	      NEEDIT[d]=TRUE;
      else
	    {
        for(d=0;d<gn2;d++)
	      NEEDIT[d]=FALSE;
        for(d=0;(d<=(band+*LengthH)/2)&&(d<gn2);d++)
          NEEDIT[d]=TRUE;
        for(d=1;(d<=(band+*LengthH)/2)&&(d<gn2);d++)
          NEEDIT[gn2-d]=TRUE;
	    }

      allocateSigma(&Sigma2,NEEDIT);
      allocateSigma(&Sigma3,NEEDIT);

      /* First the main diagonal */
      if(Sigma.diag[0]!=NULL)
	    for(j=0;j<gn;j++)
		  if(fabs(sig=Sigma.diag[0][j])>thr)
		    {
		    o1=j%2;
            iiH=j/2;
            iiG=(iiH+LengthH2-1)%(gn2);

		    for(k=0;k<LengthH2;k++)
			  {
              Sigma3.diag[0][iiH]+=sig*ProductH[2*k+o1][2*k+o1];
              Sigma2.diag[0][iiG]+=sig*ProductG[2*k+o1][2*k+o1];
              jjH=((j-2*(k+1)+*gridn)/2)%(gn2);
              jjG=(jjH+LengthH2-1)%(gn2);
		      for(l=k+1;l<LengthH2;l++)
			    { 
				dH=abs(iiH-jjH);
				dG=abs(iiG-jjG);
				if(dH==0)
                  Sigma3.diag[dH][iiH]+=2*sig*ProductH[2*k+o1][2*l+o1];
				else       
                  Sigma3.diag[dH][(iiH+jjH-dH)/2]+=sig*ProductH[2*k+o1][2*l+o1];
				if(dG==0)
                  Sigma2.diag[dG][iiG]+=2*sig*ProductG[2*k+o1][2*l+o1];
				else         
                  Sigma2.diag[dG][(iiG+jjG-dG)/2]+=sig*ProductG[2*k+o1][2*l+o1];
            
                if((--jjH)<0) jjH=gn2-1;
                if((--jjG)<0) jjG=gn2-1;
				}
              if((--iiH)<0) iiH=gn2-1;
              if((--iiG)<0) iiG=gn2-1;
		      }
            }

      /* Now  do the rest */
      for(d=1;d<gn;d++)
        if(Sigma.diag[d]!=NULL)
	      for(j=0;j<gn-d;j++)
		    if(fabs(sig=Sigma.diag[d][j])>thr)
		      {
			  o1=(d+j)%2;
              iiH=((d+j+*gridn)/2)%(gn2);
              iiG=(iiH+LengthH2-1)%(gn2);

		      for(k=0;k<LengthH2;k++)
			    {
                jjH=((j+*gridn)/2)%(gn2);
                jjG=(jjH+LengthH2-1)%(gn2);
			    o2=j%2;
		        for(l=0;l<LengthH2;l++)
				  {
				  dH=abs(iiH-jjH);
				  dG=abs(iiG-jjG);
				  if(dH==0)
                    Sigma3.diag[dH][iiH]+=2*sig*ProductH[o1][o2];
				  else       
                    Sigma3.diag[dH][(iiH+jjH-dH)/2]+=sig*ProductH[o1][o2];
				  if(dG==0)
                    Sigma2.diag[dG][iiG]+=2*sig*ProductG[o1][o2];
				  else         
                    Sigma2.diag[dG][(iiG+jjG-dG)/2]+=sig*ProductG[o1][o2];
                  if((--jjH)<0) jjH=gn2-1;
                  if((--jjG)<0) jjG=gn2-1;
                  o2+=2;
                  }
                if((--iiH)<0) iiH=gn2-1;
                if((--iiG)<0) iiG=gn2-1;
				o1+=2;
				}
              }
	  for(j=0;j<gn2;j++)
        c[zaehler++]=Sigma2.diag[0][j];
/*
      puts("Detail coefficients");
      reportSigma(&Sigma2);
      puts("Smooth coefficients");
      reportSigma(&Sigma3);
*/
      freeSigma(&Sigma);
      freeSigma(&Sigma2);
      /*createSigma(&Sigma,gn2);*/
      memcpy(&Sigma,&Sigma3,sizeof(Sigma));
  	  gn=gn2;
  	  }
	}
  else
  if(*bc==SYMMETRIC)
    {
    virtgn=gn;
    offset=0;
    while(virtgn>=2)
      {
	  if(offset%2==0)
	    gn2=(gn+1)/2+LengthH2-1;
      else
	    gn2=(gn+2)/2+LengthH2-1;
	  offset2=(offset+*LengthH-1)/2;
	  
      createSigma(&Sigma2,gn2);
      createSigma(&Sigma3,gn2);
      cleanupSigma(&Sigma);
      
      /* Compute actual bandwidth */

	  band=gn-1;
	  while((band>=0)&&(Sigma.diag[band]==NULL))
	    band--;
      if(band+2*(*LengthH)>gn)
        for(d=0;d<gn2;d++)
	      NEEDIT[d]=TRUE;
      else
	    {
        for(d=0;d<gn2;d++)
  	      NEEDIT[d]=FALSE;
        for(d=0;(d<=(band+*LengthH)/2)&&(d<gn2);d++)
          NEEDIT[d]=TRUE;
        }

      allocateSigma(&Sigma2,NEEDIT);
      allocateSigma(&Sigma3,NEEDIT);

      /* Let's start with the values that are not effected by boundary correction */
      /* First the main diagonal */
      if(Sigma.diag[0]!=NULL)
	    /*for(j=*LengthH-2+offset%2;j<gn-(*LengthH-2+offset%2);j++)*/
	    for(j=0;j<gn;j++)
		  if(fabs(sig=Sigma.diag[0][j])>thr)
		    {
		    o1=(j+offset)%2;
            iiH=(j+offset%2)/2+LengthH2-1;
			iiG=iiH;

		    for(k=0;k<LengthH2;k++)
			  {
              Sigma3.diag[0][iiH]+=sig*ProductH[2*k+o1][2*k+o1];
              Sigma2.diag[0][iiG]+=sig*ProductG[2*k+o1][2*k+o1];
              jjH=(j+offset%2)/2+LengthH2-1-(k+1);
			  jjG=jjH;
		      for(l=k+1;l<LengthH2;l++)
			    { 
				if((iiH<0)||(jjH<0))
				  puts("ERROR");
				dH=abs(iiH-jjH);
				dG=abs(iiG-jjG);
				if(dH==0)
                  Sigma3.diag[dH][iiH]+=2*sig*ProductH[2*k+o1][2*l+o1];
				else       
                  Sigma3.diag[dH][(iiH+jjH-dH)/2]+=sig*ProductH[2*k+o1][2*l+o1];
				if(dG==0)
                  Sigma2.diag[dG][iiG]+=2*sig*ProductG[2*k+o1][2*l+o1];
				else       
                  Sigma2.diag[dG][(iiG+jjG-dG)/2]+=sig*ProductG[2*k+o1][2*l+o1];
                jjH--;
                jjG--;
				}
              iiH--;
              iiG--;
		      }
            }

      /* Now do the not-effected values on the other diagonals */
      for(d=1;d<gn;d++)
        if(Sigma.diag[d]!=NULL)
	      /*for(j=*LengthH-2+offset%2;j<gn-d-(*LengthH-2+offset%2);j++)*/
	      for(j=0;j<gn-d;j++)
		    if(fabs(sig=Sigma.diag[d][j])>thr)
		      {
		      o1=(d+j+offset)%2;
              iiH=(d+j+offset%2)/2+LengthH2-1;
			  iiG=iiH;

		      for(k=0;k<LengthH2;k++)
			    {
                jjH=(j+offset%2)/2+LengthH2-1;
			    jjG=jjH;
			    o2=(j+offset)%2;
		        for(l=0;l<LengthH2;l++)
				  {
				  dH=abs(iiH-jjH);
				  dG=abs(iiG-jjG);
				  if(dH==0)
                    Sigma3.diag[dH][iiH]+=2*sig*ProductH[o1][o2];
				  else       
                    Sigma3.diag[dH][(iiH+jjH-dH)/2]+=sig*ProductH[o1][o2];
				  if(dG==0)
                    Sigma2.diag[dG][iiG]+=2*sig*ProductG[o1][o2];
				  else         
                    Sigma2.diag[dG][(iiG+jjG-dG)/2]+=sig*ProductG[o1][o2];
                  jjH--;
                  jjG--;
                  o2+=2;
                  }
				iiH--;
				iiG--;
				o1+=2;
				}
              }

      /* Finally, do the boundary values */

      for(d=0;d<gn2;d++)
        if(Sigma2.diag[d]!=NULL)
		  {
		  for(j=0;(j<gn2-d)&&(j<LengthH2-1+offset%2);j++)
		    {
			cellC=0;
			cellD=0;
			i=d+j;
			iiG=2*i-(*LengthH-2)-offset%2;
		    for(k=0;k<*LengthH;k++,iiG++)
              {
		      jjG=2*j-(*LengthH-2)-offset%2;
		      for(l=0;l<*LengthH;l++,jjG++)
			    {
                ii=iiG;
                jj=jjG;
				if(ii<0)
				  ii=-ii-1;
				if(jj<0)
				  jj=-jj-1;
				if(ii>=gn)
                  {
				  ii=2*gn-ii-1;
			      }
				if(jj>=gn)
                  {
				  jj=2*gn-jj-1;
			      }
				dd=abs(ii-jj);
                if(Sigma.diag[dd]!=NULL)
                  {
				  sig=(Sigma.diag)[dd][(ii+jj-dd)/2];
				  cellC+=sig*ProductH[k][l];
				  cellD+=sig*ProductG[k][l];
                  }
				}
              }
            Sigma2.diag[d][j]=cellD;
            Sigma3.diag[d][j]=cellC;
			}
		  for(j=gn2-d-1;(j>=0)&&(j>=gn2-d-LengthH2+1-offset%2);j--)
		    {
			cellC=0;
			cellD=0;
			i=d+j;
			iiG=2*i-(*LengthH-2)-offset%2;
		    for(k=0;k<*LengthH;k++,iiG++)
              {
		      jjG=2*j-(*LengthH-2)-offset%2;
		      for(l=0;l<*LengthH;l++,jjG++)
			    {
                ii=iiG;
                jj=jjG;
				if(ii<0)
				  ii=-ii-1;
				if(jj<0)
				  jj=-jj-1;
				if(ii>=gn)
                  {
				  ii=2*gn-ii-1;
			      }
				if(jj>=gn)
                  {
				  jj=2*gn-jj-1;
			      }
				dd=abs(ii-jj);
                if(Sigma.diag[dd]!=NULL)
                  {
				  sig=(Sigma.diag)[dd][(ii+jj-dd)/2];
				  cellC+=sig*ProductH[k][l];
				  cellD+=sig*ProductG[k][l];
                  }
				}
              }
            Sigma2.diag[d][j]=cellD;
            Sigma3.diag[d][j]=cellC;
			}
		  }

	  for(j=0;j<gn2;j++)
        c[zaehler++]=Sigma2.diag[0][j];
/*
      puts("Detail coefficients");
      reportSigma(&Sigma2);
      puts("Smooth coefficients");
      reportSigma(&Sigma3);
*/
      freeSigma(&Sigma);
      freeSigma(&Sigma2);
      /*createSigma(&Sigma,gn2);*/
      memcpy(&Sigma,&Sigma3,sizeof(Sigma));

      gn=gn2;
	  offset=offset2;
      virtgn=virtgn/2;
      }
	}
freeSigma(&Sigma);
free(NEEDIT);
  }
