#' Create fancy ticks like in cool engineering diagrams
#'
#' Ticks of different sizes.  Big ticks are where \code{\link{pretty}}
#' wants to put them.  Medium ticks are on the halves, and small ticks
#' are on the fifths of the halves.  This function is a bit
#' hit-or-miss, since \code{pretty} is so deep, so try changing
#' \code{n} to get the ticks you want.
#'
#' @section Utilities:
#' Uses the internal function
#' \code{is.wholenumber}, which does what you expect.
#'
#' @aliases is.wholenumber
#'
#' @param x values for which ticks are required (could be an \code{xlim})
#' @param n resolution parameter, passed to \code{\link{pretty}}
#' @param level how many levels of ticks?
#'
#' @return A list of \code{level} components, where each components
#' contains that level's ticks, as a numerical vector (which may have
#' length zero). Also an \code{xlim} attribute.
#'
#' @author Jonathan Rougier <j.c.rougier@@bristol.ac.uk>
#' @export
#' @seealso \code{\link{axis.paranomo}}, which calls \code{fancyticks}
#'
#' @examples
#'
#' fancyticks(c(85, 220), 10)

fancyticks <- function(x, n = 7, level = 3) {

    ticks <- rep(list(numeric()), level)

    tt <- pretty(x, n = n)
    k <- length(tt)
    if (k == 1) {
        warning("No interesting ticks, increase \'n\'")
        return(ticks)
    }

    ## padding is helpful here and below
    
    dd <- diff(tt[1:2])
    tt <- c(tt[1] - dd, tt, tt[k] + dd)
    k <- k + 2

    ticks[[1]] <- tt
    
    ## ticks on the halves
    
    if (level > 1) {
        if (is.wholenumber(log10(dd))) {
            dd <- dd / 2
            ticks[[2]] <- seq(tt[1] - dd, tt[k] + dd, by = dd)
        }
    }
    
    ## ticks on the fifths
    
    if (level > 2) {
        tt <- ticks[[2]]
        k <- length(tt)
        if (k == 0) {
            tt <- ticks[[1]]
            k <- length(tt)
        }
        if (k > 1) {
            dd <- diff(tt[1:2])
            if (is.wholenumber(log10(2 * dd))) {
                dd <- dd / 5
                ticks[[3]] <- seq(tt[1] - dd, tt[k] + dd, by = dd)
            }
        }
    }

    ## trim and remove duplicates

    xlim <- range(x)
    ticks <- lapply(ticks, function(x) x[x >= xlim[1] & x <= xlim[2]])

    if (level > 1)
        ticks[[2]] <- setdiff(ticks[[2]], ticks[[1]])

    if (level > 2) {
        uu <- union(ticks[[1]], ticks[[2]])
        ticks[[3]] <- setdiff(ticks[[3]], uu)
    }

    attr(ticks, "xlim") <- range(unlist(ticks))
    ticks
}

## relaxed wholenumber function

is.wholenumber <- function(x, tol = 1e-6)
    is.finite(x) & abs(x - round(x)) < tol
