#' Parallel scale nomogram
#'
#' Return the functions necessary to construct a parallel-scale
#' nomogram of three quantities, which are related as \deqn{f(x) +
#' g(y) = h(z).}
#'
#' @param f,g,h monotonic functions taking vector arguments
#' @param xlim,ylim,zlim lower and upper limits of the three quantites
#' @param xmid,ymid Centre-points of x- and y-axes
#' @param plot plot the result?
#' @param add add to existing plot?
#' @param ... additional arguments to \code{plot.paranomo}
#'
#' @details Specifying \code{xmid} and \code{ymid} is useful for
#' aligning up the x- and y-axes so that typical (middle) evaluations
#' occur with fairly flat lines.  Otherwise, the x- and y-axes will be
#' aligned at the bottom and top.
#'
#' @return An object of class \code{paranomo}, which is a list of
#' three functions, \code{xput}, \code{yput}, and \code{zput}, which
#' share a common environment (plus other stuff).  \code{xput}, for
#' example, takes a vector of \eqn{x} values and returns a
#' corresponding matrix of horizontal and vertical ordinates on a
#' standardised frame, which is approximately \eqn{[-1, 1] \times [-1,
#' 1]}{[-1, 1] x [-1, 1]}.
#'
#' @author Jonathan Rougier <j.c.rougier@@bristol.ac.uk>
#' @export
#' @seealso \code{\link{axis.paranomo}} and \code{\link{plot.paranomo}} for plotting, and examples.
#' @references R. Doerfler, 2009, The lost art of nomography, UMAP Journal, 30(4), 457-493.

paranomo <- function(f, g, h, xlim, ylim, zlim, xmid = NULL, ymid = NULL,
                     plot = TRUE, add = FALSE, ...) {

    stopifnot(is.function(f), is.function(g), is.function(h),
              diff(xlim) > 0, diff(ylim) > 0, diff(zlim) > 0,
              is.logical(plot), is.logical(add))

    ## retain labels for later

    labels.default <- c(deparse(substitute(f)),
                       deparse(substitute(g)),
                       deparse(substitute(h)))

    ## sort out the xmid and ymid values

    flim <- f(xlim)
    if (is.null(xmid)) {
        x <- seq(from = xlim[1], to = xlim[2], length.out = 51)
        xmid <- approx(f(x), x, mean(flim))$y
    }

    glim <- g(ylim)
    if (is.null(ymid)) {
        y <- seq(from = ylim[1], to = ylim[2], length.out = 51)
        ymid <- approx(g(y), y, mean(glim))$y
    }
    
    ## construct for a frame that is [-1, 1] by [-1, 1]

    fmid <- f(xmid)
    gmid <- g(ymid)
    hmid <- fmid + gmid

    alpha <- 2 / abs(diff(flim))
    beta <- 2 / abs(diff(glim))
    
    u <- alpha / (alpha + beta)
    gamma <- beta * u

    ## create put functions

    xput <- function(x) cbind(0, alpha * (f(x) - fmid))
    yput <- function(y) cbind(1, beta *  (g(y) - gmid))
    zput <- function(z) cbind(u, gamma * (h(z) - hmid))

    ## will return as a list; vlim is for plot.window()

    v <- c(xput(xlim)[, 2], yput(ylim)[, 2], zput(zlim)[, 2])

    pobj <- list(xput = xput, yput = yput, zput = zput,
                 xlim = xlim, ylim = ylim, zlim = zlim,
                 u = u, vlim = range(v),
                 labels.default = labels.default)
    class(pobj) <- "paranomo"

    ## plot if necessary

    if (plot)
        plot(pobj, add = add, ...)

    ## and return

    pobj
}
