#' @name plot.paranomo
#' @rdname plot.paranomo
#' @title Plotting for parallel-axis nomograms
#'
#' @description Plotting functions for parallel-axis nomogram object
#' of class \code{paranomo}, created with the \code{paranomo}
#' function.  \code{plot.paranomo} plots the whole diagram, while
#' \code{axis.paranomo} just plots one axis.
#'
#' @seealso \code{\link{paranomo}} for creating a parallel-axis nomogram object.
#' 
#' @examples
#' ## ## Fig 3 in Doerfler (2009)
#' 
#' D <- function(D) 0.68 * log(1.2 * D + 0.47)
#' T <- function(T) (3/2) * log(0.91 * T)
#' N <- function(N) log(N)
#' foo <- paranomo(D, T, N,
#'                 xlim = c(1, 8), ylim = c(1, 2), zlim = c(1, 11.5))
#'  
#' Dval <- 5; Tval <- 1.43; Nval <-exp(D(Dval) + T(Tval))
#' lines(rbind(foo$xput(Dval), foo$yput(Tval)), lty = 2)
#' points(rbind(foo$xput(Dval),
#'              foo$yput(Tval),
#'              foo$zput(Nval)), pch = 20, cex = 1.2)
#'
#' ## different colour, or colours?
#'
#' plot(foo, col = "blue")
#' plot(foo, col = c("red", "gold", "green"))
#'
#' ## fool around with n to get the ticks right
#' 
#' plot(foo, n = 7)
#' plot(foo, n = c(7, 10, 10))
#'
#' ## include midpoints for different alignment
#' 
#' foo <- paranomo(D, T, N,
#'                 xlim = c(1, 8), ylim = c(1, 2), zlim = c(1, 11.5),
#'                 xmid = 2, ymid = 1.4)
#'
#' ## ## Nomogram of Body Mass Index (BMI), has decreasing y axis
#'
#' Weight <- function(weight) log(weight) # in kg
#' Height <- function(height) -2 * log(height) # in m
#' BMI <- function(BMI) log(BMI)
#'
#' bmi <- paranomo(Weight, Height, BMI,
#'                 xlim = c(40, 160), ylim = c(1.5, 2), zlim = c(15, 40),
#'                 n = c(10, 10, 7))
#' lines(rbind(bmi$xput(75), bmi$yput(1.82)), lty = 2) # me, roughly
#'
NULL

#' @rdname plot.paranomo
#'
#' @param x,pobj object of class \code{paranomo}
#' @param labels axis labels
#' @param add add to existing plot?
#' @param ... arguments to \code{axis.paranomo}
#'
#' @export

plot.paranomo <- function(x, labels, add = FALSE, ...) {

    pobj <- x
    stopifnot(inherits(pobj, "paranomo"))
    if (missing(labels))
        labels <- pobj$labels.default

    ## set up the plotting window
    
    if (!add) {
        plot.new()
        plot.window(xlim = c(0, 1), ylim = pobj$vlim)
    }

    ## plot all three axes, handling parameters (tcl is a bit different)

    args <- list(...)
    for (i in 1L:3) {
        nargs <- lapply(args, function(x) if (length(x) == 1) x else x[i])
        nargs <- c(list(pobj = pobj, side = c("x", "y", "z")[i], label = labels[i]), nargs)
        nargs$tcl <- args$tcl # may be NULL
        do.call("axis.paranomo", nargs)
    }
    invisible(NULL)
}

#' @rdname plot.paranomo
#'
#' @param side axis identifier
#' @param label axis label
#' @param col axis colours
#' @param n,tick.level arguments to \code{\link{fancyticks}}
#' @param tcl tick lengths (vector the same length as \code{tick.level})
#' @param lwd axis line width (ticks are \code{lwd = 1})
#' @param cex.axis axis numbers expansion
#' @param offset axis label placement (above axis)
#' @param cex.lab axis label expansion
#' @param font.lab axis label font
#' @param col.lab axis label colours
#'
#' @details \code{axis.paranomo} uses \code{\link{fancyticks}} to
#' compute the ticks for each axis.  Fiddle with \code{n} (and
#' possibly the limits in \code{paranomo}) to get different ticks.
#' All of the axis parameters provided to \code{plot.paranomo} can be
#' three-vectors for individual axis tuning.
#'
#' @export

#### put one axis onto a parallel-axis nomogram

axis.paranomo <- function(pobj,
                          side = c("x", "y", "z"), label, col = 1,
                          n = 7, tick.level = 3, tcl = -c(0.75, 0.5, 0.25),
                          lwd = 2, cex.axis = 0.8,
                          offset = 1, cex.lab = 1, font.lab = 1, col.lab = col) {

    stopifnot(inherits(pobj, "paranomo"))
    side <- match.arg(side)

    put <- pobj[[sprintf("%sput", side)]]
    lim <- pobj[[sprintf("%slim", side)]]
    if (missing(label))
        label <- pobj$labels.default[match(side, c("x", "y", "z"))]

    pos <- put(lim)[1, 1]
    mm <- match(side, c("x", "y", "z"))
    side <- c(2, 4, ifelse(pos <= 0.5, 4, 2))[mm]

    ## sort out the ticks

    ticks <- fancyticks(lim, n = n, level = tick.level)
    
    ## now draw the numbers and the ticks

    xy <- put(tt <- ticks[[1]])
    
    axis(side, xy[, 2], labels = tt, pos = pos,
         las = 1, cex.axis = cex.axis, col.axis = col,
         lwd = 0, lwd.ticks = 1, tcl = tcl[1], col.ticks = col)

    ## additional ticks may be useful

    if (tick.level > 1 && length(tt <- ticks[[2]]) > 0) {
        axis(side, put(tt)[, 2], labels = FALSE, pos = pos,
             lwd = 0, lwd.ticks = 1, tcl = tcl[2], col.ticks = col)
    }
    
    if (tick.level > 2 && length(tt <- ticks[[3]]) > 0) {
        axis(side, put(tt)[, 2], labels = FALSE, pos = pos,
             lwd = 0, lwd.ticks = 1, tcl = tcl[3], col.ticks = col)
    }

    ## complete with an axis and label

    axis(side, put(attr(ticks, "xlim"))[, 2], labels = FALSE, pos = pos,
         lwd = lwd, lwd.ticks = 0, col = col, lend = "square")
    
    text(pos, max(xy[, 2]), label, pos = 3,
         offset = offset, cex = cex.lab, font = font.lab, xpd = NA,
         col = col.lab)


    invisible(NULL)
}
